/*------------ monitor.c --------*/
/* monitor command */

#include <stdio.h>
#include <signal.h>
#include <sys/pstate.h>
#include <sys/param.h>
#include <sys/times.h>
#include <a.out.h>

#define HYPHEN          '-'
#define OUTPUT          'o'
#define INPUT           'i'
#define MILLISECS       'm'
#define MICROSECS       'u'
#define DEF_TSLICE      0x0000000001000000L     /* 1/250 sec */
#define COLON           ':'
#define EOS             '\0'
#define SLASH           '/'
#define PATH            "PATH="
#define hibit(a)        ((a) & 0x80)

struct  ftn     {
	struct  nlist   *f_nlist;       /* symbol table entry for this ftn */
	int     f_hits;                 /* number of hits in this function */
};

struct  syms    {
	struct  syms    *s_next;        /* ptr to next */
	struct  nlist   s_nlist;        /* pseudo symbol table entry */
	int     s_hits;
};


char    usage[] = "Usage: %s [-m ms] [-u us] [-o fn1] [-i fn2] program [args ...]\n";
char    monout[] = "mon.out";
struct  ftn     *funcs;                 /* function bucket array */
float   usrtime = 0;

main(argc, argv, envp)
int     argc;
char    **argv;
char    **envp;
{
        int     i, n;
        int     p;              /* process id of child */
        int     pid;            /* process id of child */
	int     status;         /* returned from wait() */
	int     nfunc;          /* number of functions */
	int     compar1();
	char    *cmdname;
	char    *fn;            /* output filename */
	FILE    *mon;           /* output file descriptor */
	struct  ftn     *ftnp;  /* ptr within funcs[] */
	struct  pstate  state;  /* process state buffer */
	long    tslice;         /* time slice */
	struct  tms     before;
	struct  tms     after;


	nfunc = 0;
	tslice = DEF_TSLICE;
	cmdname = *argv;
        if (argc <= 1) {
                fprintf(stderr, usage, cmdname);
		exit(1);
	}

	fn = NULL;
        argv++;
	argc--;
	while (**argv == HYPHEN) {
		argc--;
                if (--argc <= 0) {
                        fprintf(stderr, usage, cmdname);
                        exit(1);
                }
		switch (*(++*argv)) {
case MILLISECS:
                        argv++;
			tslice = atoi(*argv++)*1000 << 12;
			break;
case MICROSECS:
                        argv++;
			tslice = atoi(*argv++) << 12;
			break;
case OUTPUT:
                        argv++;
                        fn = *argv++;
			break;
case INPUT:
                        argv++;
			if (!fn)
				fn = *argv;
                        nfunc = rdfile(*argv++);
			break;
default:
                        fprintf(stderr, usage, cmdname);
                        exit(1);
		}
	}

	if (!fn)
	        fn = monout;
	if (!nfunc)
	        nfunc = rdsym(*argv, envp);
	qsort(funcs, nfunc, sizeof(*ftnp), compar1);
        signal(SIGINTR, SIG_IGN);

        if ((p = fork()) == -1) {
                fprintf(stderr, "Fork failed, try again.\n");
                exit(1);
        }
        if (p == 0) {
                signal(SIGINTR, SIG_DFL);
                sigtrace(0, SIGEXEC, 1);
                execvp(*argv, argv, 0);
                fprintf(stderr, "Execvp failed.\n");
                exit(1);
        }
        if ((mon = fopen(fn, "a")) == NULL) {
                fprintf(stderr, "%s: Can't write %s\n", cmdname, fn);
		exit(1);
        }
	fclose(mon);
        /*
         *  Monitor the process
         */
        if (((pid = wait(&status)) == p) && ((status & 0177) == 0177)) {
                ptrace(p, 0, &state);
                if (state.ps_sig != SIGEXEC) {
			fprintf(stderr, "%d: Not SIGEXEC.\n", state.ps_sig);
                        exit(1);
		}
		times(&before);
                state.ps_tslice = tslice;
                state.ps_sig = 0;
                sigtrace(p, SIGSLICE, 1);
                ptrace(p, 1, &state);
	} else {
		fprintf(stderr, "%s: Unexpected wait: %o\n", cmdname, status);
		exit(1);
	}

        while((wait(&status) == p) && ((status & 0177) == 0177)) {
                ptrace(p, 0, &state);
                if (state.ps_sig == SIGSLICE) {
			if ((n = srch(funcs, nfunc, state.ps_pc)) != -1)
                                funcs[n].f_hits++;
		}
                /*
                 *  If the program exec'ed itself, then we're done
                 */
		else if (state.ps_sig == SIGEXEC) {
                        state.ps_tslice = tslice;
                        state.ps_sig = 0;
                        ptrace(p, 1, &state);
			sigtrace(p, SIGSLICE, 0);
			break;
		}
		else {
			fprintf(stderr, "%d: Bad signal.\n", state.ps_sig);
                        exit(1);
		}
                state.ps_tslice = tslice;
                state.ps_sig = 0;
                ptrace(p, 1, &state);
        }

	times(&after);
	usrtime += (float) (after.tms_cutime - before.tms_cutime)/1000000;

        if ((mon = fopen(fn, "w")) == NULL) {
                fprintf(stderr, "%s: Can't write %s\n", cmdname, fn);
		exit(1);
        }

	fprintf(mon, "%f\n",  usrtime);
	for (ftnp = funcs; ftnp < &funcs[nfunc]; ftnp++)
                fprintf(mon, "%s %x %d\n", ftnp->f_nlist->n_name,
			ftnp->f_nlist->n_value, ftnp->f_hits);
	exit((status >> 8) & 0xff);
}

/* rdsym - read symbol table from an executable program
*  returns:     the number of symbols
*/
rdsym(fn, envp)
char    *fn;
char    **envp;
{
	FILE    *fd;            /* file descriptor */
        int     offset;         /* seek offset for symbol table */
	int     n;
	struct  ftn     *ftnp;  /* ftn pointer */
	struct  exec    hdr;    /* header of executable program */
	struct  nlist   *symtab;/* beginning of symbol table */
	struct  nlist   *sym;   /* current symbol table entry */
	int     nsym;           /* number of symbols */
	char    pathname[64];   /* full pathname to the program */
	char    *path;          /* ptr to environment path */
	char    *p;

        while (*envp) {
                if (strncmp(*envp, PATH, sizeof(PATH)-1) == 0)
                        break;
                envp++;
        }
	path = *envp + sizeof(PATH)-1;
	for (;;) {
                if (*path == EOS) {
                        fprintf(stderr, "%s: Unable to open\n", fn);
                        exit(1);
	        }
                for (p = path; *p && *p != COLON; p++)
                        ;
		if ((n = p - path) > 0) {
                        strncpy(pathname, path, n);
		        pathname[n++] = SLASH;
		}
                strcpy(&pathname[n], fn);
                if ((fd = fopen(pathname, "r")) != NULL)
                        break;
	        path = ++p;
	}
        /*
         * Read the symbol table
         */
	fread(&hdr, sizeof(hdr), 1, fd);
        nsym = hdr.a_syms/sizeof(*sym);
	if (nsym <= 0) {
		fprintf(stderr, "%s: No symbol table\n", fn);
		exit(1);
	}
	symtab = (struct nlist *) malloc(hdr.a_syms);
	offset = sizeof(hdr) + hdr.a_text + hdr.a_data + hdr.a_reloc;
	if (fseek(fd, offset, 0) == -1) {
		fprintf(stderr, "%s: Unable to read symbol table\n", fn);
		exit(1);
	}
	if (fread(symtab, sizeof(*sym), nsym, fd) != nsym) {
		fprintf(stderr, "%s: Unable to read symbol table\n", fn);
		exit(1);
	}
        /*
         * Create the function bucket array
         */
	funcs = (struct ftn *) malloc(nsym*sizeof(*funcs));
	ftnp = funcs;
	for (sym = symtab; sym < &symtab[nsym]; sym++) {
		if ((sym->n_type & N_TYPE) == N_TEXT) {
			if (!hibit(sym->n_name[0])) {
                                ftnp->f_nlist = sym;
                                ftnp++;
			}
		}
        }
	fclose(fd);
	return(ftnp - funcs);
}

/* rdfile - initialize function bucket array from existing monitor file
*  returns:  number of functions,
*            0, if unable to read the file
*/
rdfile(fn)
char    *fn;            /* file name */
{
	FILE    *fd;            /* file descriptor */
	struct  syms    *symbase;
	struct  syms    *sym;
	struct  ftn     *ftnp;
	int     nsym;

        if ((fd = fopen(fn, "r")) == NULL)
                return(0);
/*
 * Read the monitor file
 */
	fscanf(fd, "%f", &usrtime);
	symbase = (struct syms *) malloc(sizeof(*sym));
	sym = symbase;
	nsym = 0;
	while (fscanf(fd, "%s %x %d", sym->s_nlist.n_name,
		          &sym->s_nlist.n_value, &sym->s_hits) != EOF) {
		nsym++;
		sym->s_next = (struct syms *) malloc(sizeof(*sym));
		sym = sym->s_next;
		sym->s_next = NULL;
	}
/*
* Create the function bucket array
*/
	funcs = (struct ftn *) malloc((nsym+1)*sizeof(*ftnp));
	ftnp = funcs;
	for (sym = symbase; sym; sym = sym->s_next) {
                ftnp->f_nlist = &sym->s_nlist;
		ftnp->f_hits = sym->s_hits;
                ftnp++;
        }
	fclose(fd);
	return(nsym);
}

compar1(x, y)
struct  ftn     *x;
struct  ftn     *y;
{
	return(x->f_nlist->n_value - y->f_nlist->n_value);
}


/* srch - binary search
*  returns: an index to the found item, or
*            -1 if the item not found.
*/
srch(funcs, i, psw)
struct  ftn     *funcs;
int     i;
int     psw;
{
	int     lo, hi, n;
	int     rc;

        lo = 0;
        hi = i - 1;
        while (lo <= hi) {
                n = (lo + hi)/2;
		rc = psw - funcs[n].f_nlist->n_value;
		if (rc == 0)
                        return(n);
                else if (rc < 0) {
			if (n == 0)
				return(-1);
			if (psw - funcs[n-1].f_nlist->n_value > 0)
				return(n-1);
                        hi = n - 1;
		} else {
			if (n >= i)
				return(n);
			if (psw - funcs[n+1].f_nlist->n_value < 0)
				return(n);
                        lo = n + 1;
		}
        }
	return(-1);
}
