/* SEND command
 *
 * Synopsis : send <-lqfrtke> <filename ...>
 *
 * The flags are as follows:
 *
 *   -l     List card images on standard output.
 *   -q     Do not submit jcl to MVS.
 *   -f     Fold lower case to upper.
 *   -r     suppress translation from ascii to ebcdic.
 *   -t     Trace generation of jcl, listing all input before
 *          and after substitution.
 *   -k     Turn off keyword substitution for non-control lines.
 *   -i     Don't interpret control lines; treat them as text.
 *          Note: Once this flag is set it cannot be reset.
 *   -s     Make keyword substitutions before interpreting
 *          control lines.
 *   -e     Erase all previous keyword definitions.
 *   -d     Display current keyword definitions.
 *   -uUSER User to recieve job output (if returned).  Default
 *          is current user.
 *   -cNAME Linkname for RSCS to send job to.
 *
 * Control lines are input lines that begin with a "~".  Immediately
 * following the "~" can be one of the following commands:
 *
 *   -flags             Set the given flags as described above.
 *   +flags             Reset the given flags.
 *   -:prompt           Print prompt if standard input is a terminal.
 *   +:prompt           Print prompt no matter what.
 *   !command           Execute the specified UNIX command.  Standard
 *                      output of command is read as input to the send
 *                      command.
 *   ~comment           Ignore this comment.
 *   =:keyword;default  Read line from standard input and assign
 *                      string to keyword.  Use default if only '\n'
 *                      is recieved.
 *   keyword=string     Define keyword as the given string.
 *   filename           Start reading from given filename.  Returns
 *                      to following line at EOF of filename (note:
 *                      this specification may be nested).
 *   *filename          "Include" the given filename.  Do not do any
 *                      modifications to the data, and do not check for
 *                      command lines.
 *   ?filename          "Include" the given filename.  Do not do any
 *                      modifications to the data, and block it into
 *                      80 byte records.  No checking of command lines
 *                      is performed.
 *   ==keyword;n        Skip next n lines if keyword is null.  The ;n
 *                      may be omitted, and n defaults to 1.
 */
 
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>

extern char atetab[];

/* flags */
int lflag, qflag, fflag;
int rflag, tflag, kflag;
int iflag, sflag;

#define  MAXKEYS   200
#define  MAXNEST    10
#define  ESCAPE    '~'
#define  MTY       060044
#define  CTY       020000
#define  LNL       302
#define  NEWLINE   '\n'
#define  MAXLINE    80
#define  MAXKEYL    40
#define  MAXSUB    100
#define  OUTSYS    "rscs"
#define  INTR      2

struct {
       char  *keyword;           /* pointer to keyword string        */
       char  *subst;             /* pointer to substitution string   */
       int   keylen;             /* length of keywd for reverse sort */
} keys[MAXKEYS];

int  numkeys;                         /* number of keys defined now     */
char line[LNL];                       /* buffer for input line          */
char wrkbuf[LNL];                     /* working buffer for input line  */
char tempfil[] = "/usr/spool/sendXXXXX"; /* output file for created jcl */
int  errflg;                          /* incremented if error detected  */
int  tin;                             /* 1 primary input is TTY, else 0 */
int  skipline;                        /* number of lines to skip if an  */
				      /* == control line is found       */
char complex[20] = "complex1";        /* rscs routing link name         */
char auuser[9];                       /* Au user to recieve any output  */
FILE *outfile;                        /* output file descriptor         */
FILE *maketemp();

extern int cleanup();

char lowtup[256] =
{  0000,0001,0002,0003,0004,0005,0006,0007,0010,0011,0012,
   0013,0014,0015,0016,0017,0020,0021,0022,0023,0024,0025,
   0026,0027,0030,0031,0032,0033,0034,0035,0036,0037,0040,
   0041,0042,0043,0044,0045,0046,0047,0050,0051,0052,0053,
   0054,0055,0056,0057,0060,0061,0062,0063,0064,0065,0066,
   0067,0070,0071,0072,0073,0074,0075,0076,0077,0100,0101,
   0102,0103,0104,0105,0106,0107,0110,0111,0112,0113,0114,
   0115,0116,0117,0120,0121,0122,0123,0124,0125,0126,0127,
   0130,0131,0132,0133,0134,0135,0136,0137,0140,0101,0102,
   0103,0104,0105,0106,0107,0110,0111,0112,0113,0114,0115,
   0116,0117,0120,0121,0122,0123,0124,0125,0126,0127,0130,
   0131,0132,0173,0174,0175,0176,0177,0200,0201,0202,0203,
   0204,0205,0206,0207,0210,0211,0212,0213,0214,0215,0216,
   0217,0220,0221,0222,0223,0224,0225,0226,0227,0230,0231,
   0232,0233,0234,0235,0236,0237,0240,0241,0242,0243,0244,
   0245,0246,0247,0250,0251,0252,0253,0254,0255,0256,0257,
   0260,0261,0262,0263,0264,0265,0266,0267,0270,0271,0272,
   0273,0274,0275,0276,0277,0300,0301,0302,0303,0304,0305,
   0306,0307,0310,0311,0312,0313,0314,0315,0316,0317,0320,
   0321,0322,0323,0324,0325,0326,0327,0330,0331,0332,0333,
   0334,0335,0336,0337,0340,0341,0342,0343,0344,0345,0346,
   0347,0350,0351,0352,0353,0354,0355,0356,0357,0360,0361,
   0362,0363,0364,0365,0366,0367,0370,0371,0372,0373,0374,
   0375,0376,0377   };

char *getlogin();
char *sname();
char *malloc();
 
main(argc, argv)
int argc;
char **argv;
{
       FILE *infile;

       setbuf(stdout,(char *)NULL);
       strcpy(auuser, getlogin());           /* init user to get data */
       if (strcmp("gath",sname(*argv))==0)    /* just gather?          */
	      qflag=lflag=1;
       argv++;
       while(--argc > 0 && **argv=='-') {
	      setflags(*argv);               /* setup flags           */
	      argv++;
       }
       outfile = maketemp();                 /* setup output file     */
       if (((unsigned)signal(INTR, SIG_IGN) & 01)==0)
	      signal(INTR,cleanup);
       tin = terminp();                      /* setup tty flag        */
       if (argc<=0)
	      infile = stdin;                /* use standard input    */
       else
	      infile = fopen(*argv, "r");    /* use given filename    */
       do {
	      if (infile == NULL) {
		     fprintf(stderr, "Cannot open %s\n", *argv);
		     errflg++;
		     goto goon;
	      }
	      process(infile);
	      if (infile != stdin)
		     fclose(infile);
goon:         if (--argc > 0)
		     infile = fopen(*++argv, "r");
       } while(argc > 0);
       fclose(outfile);
       if (!qflag) submit();
       unlink(tempfil);
       exit(0);
}
 
/* process an input file of send stuff */

process(file)
FILE *file;
{
       char *iobuf;
       int len;

       iobuf = malloc(BUFSIZ);
       setbuf(file, iobuf);
       while((len=getline(file)) > 0) {
	      if (tflag) fputs(line, stderr);
	      if (skipline) {
		     skipline--;
		     continue;
	      }
	      if (line[0]==ESCAPE && !iflag) {
		     if (len>2)
			    line[len-1] = NULL;  /* remove end c/r  */
		     if (sflag && numkeys)
			    mtchkeys();            /* keyword subst.  */
		     docontrol();
	      }
	      else  {
		     donormal();
	      }
       }
       free(iobuf);
}
 
/* get the next line and return the line length */

getline(file)
FILE *file;
{
       register int   c;
       register char *l;
       int len;

       l = &line[0];
       len = 0;
       while((c=getc(file)) != EOF && len<LNL-2) {
	      len++;
	      if (c == NEWLINE) {
		  *l++ = '\0';
	          break;
	      } else
	          *l++ = c;
       }
       if (len == 0)
	      return(-1);  /* hit end of file */
       *l = NULL;
       return(len);
}
 
/*  Set flags for SEND command. */

setflags(flags)
char *flags;
{
    int set;

    if (*flags == '-')
	 set = 1;
    else
	 set = 0;
    while(*++flags)
	 switch(*flags)  {
	 case 'l':
		  lflag = set;
		  continue;
	 case 'q':
		  qflag = set;
		  continue;
	 case 'f':
		  fflag = set;
		  continue;
	 case 'x':
	 case 'r':
		  rflag = set;
		  continue;
	 case 't':
		  tflag = set;
		  continue;
	 case 'k':
		  kflag = set;
		  continue;
	 case 'i':
		  iflag = set;
		  continue;
	 case 's':
		  sflag = set;
		  continue;
	 case 'e':
		  eraseky();
		  continue;
	 case 'd':
		  showkys();
		  continue;
	 case 'u':
		  strcpy(auuser, ++flags);  /* copy it all and return */
		  return;
	 case 'c':
		  strcpy(complex, ++flags); /* likewise               */
		  return;
	 default:
		  fprintf(stderr, "Invalid flag: %c\n",*flags);
		  errflg++;
		  continue;
	 }

    return;
}
 
/* type current keyword settings on the error output */

showkys()
{
       register i;

       for(i=0; i<numkeys; i++)
	      fprintf(stderr, "\"%s\"=\"%s\"\n", keys[i].keyword,
		      keys[i].subst);
}


/*  'maketemp' creates a temporary file which will contain the collected
 *  run stream;  returns the file descriptor of the file.
 */

FILE *maketemp()
{
    FILE *fd;
    int  filedes;

    mktemp(tempfil);  /* call mktemp to add PID to temp filename */
    filedes = creat(tempfil,0644);
    fd = fdopen(filedes, "w");
    return(fd);
}


/*  'eraseky' resets the global variable, numkeys (which reflects the
 *  total active number of keywords) to zero, and releases all core
 *  which has been dynamically allocated to accommodate keyword
 *  definitions and their substitution values.
 */

eraseky()
{
       register i;

       for(i=0; i<numkeys; i++) {
	      free(keys[i].keyword);
	      free(keys[i].subst);
       }
       numkeys = 0;
}
 
/* process a control line */

docontrol()
{
       switch(line[1]) {
       case '~':
	      break;
       case '!':
	      callunix(&line[2]);
	      break;
       case '-':
	      if (line[2]==':') {
		     if (tin)
			    prompt(&line[3]);
	      }
	      else
		     setflags(&line[1]);
	      break;
       case '+':
	      if (line[2]==':')
		     prompt(&line[3]);
	      else
		     setflags(&line[1]);
	      break;
       case '=':
	      if (line[2] == ':')
		     readkey(&line[3]);
	      else if (line[2] == '=')
		     condition(&line[3]);
	      else {
		     fprintf(stderr, "Invalid send control: %s\n", line);
		     errflg++;
	      }
	      break;
       case '*':
	      include(&line[2]);
	      break;
       case '?':
	      mod80(&line[2]);
	      break;
       default:
	      if (any(&line[1], "="))
		     assnkey(&line[1]);
	      else
		     dofile(&line[1]);
       }
}
 

callunix(cmd) char *cmd;
{
       int pd[2], i, try, m;

       if (pipe(pd)<0) {
	      fprintf(stderr,"Cannot create pipe for command: %s\n", cmd);
	      errflg++;
	      return;
       }
       for (try=0; try<10; try++) {
	      if ((i=fork())>=0) break;
	      sleep(2);
       }
       if (i<0) {
	      fprintf(stderr, "Cannot fork for command: %s\n", cmd);
	      errflg++;
	      return;
       }
       if (i==0) {        /* child */
	      close(1);
	      dup(pd[1]);
	      execl("/bin/sh","sh","-c",cmd,0);
	      exit(040);
       }
       close(pd[1]);     /* parent */
       process(fdopen(pd[0],"r"));   /* process piped file */
       while(i != wait(&m));
       return;
}
 
terminp()  /* return 1 if primary input is terminal, else 0 */
{
       struct stat sb;

       if (fstat(fileno(stdin), &sb)<0) {
	      fprintf(stderr, "send primary input file not open.\n");
	      cleanup();
       }
       if((MTY&sb.st_mode)==CTY)
	      return(1);
       else
	      return(0);
}

prompt(s) char *s;  /* write s out to terminal */
{
       register char *t;

       if (*s == '"') {
	      *s++;
	      t = s;
	      while(*t && *t != '"') *t++;
	      *t = NULL;
       }
       fputs(s, stderr);
}

any(s,t) char *s, *t;   /* return 1 if any of chars in t exist in s   */
{
       register char *u;

       while(*t) {
	      u = s;
	      while(*u)
		     if (*u++ == *t) return(1);
	      *t++;
       }
       return(0);
}
 
readkey(s) char *s; /* get keyword substitute value from std. input */
{
       char key[MAXKEYL], sub[MAXSUB], def[MAXSUB];
       register char *p, *m;
       int loc, c;

       loc = token(s, key, 0, "; ", MAXKEYL);
       token(s, def, loc, "; ", MAXSUB);
       p = sub;
       m = &sub[MAXSUB];
       while((c = getchar())!=EOF && c != NEWLINE) {
	      if (p<m) *p++ = c;
       }
       if (sub[0] == NEWLINE)
	      addkey(key, def);          /* use default */
       else {
	      *++p = NULL;
	      token(sub, sub, 0, "\n ", MAXSUB);
	      addkey(key, sub);          /* use input string */
       }
}
 
/*
 * scan string s starting at location loc until a character
 * in delim is found.  Put found field in r.  Delim characters
 * are ignored if the given string is in double quotes.
 */

token(s,r,loc,delim,max) char *s, *r, *delim; int loc, max;
{
       int cnt;

       cnt = 1;
       for(; s[loc]; loc++) {  /* skip leading delimiters */
	      if (!exist(delim, s[loc]))
		     break;
       }
       if (s[loc] == '"') {
	      loc++;
	      delim = "\"";             /* change delimiter string */
       }
       for(; s[loc]; loc++) {
	      if(exist(delim, s[loc]))
		     break;
	      if (++cnt < max) *r++ = s[loc];
       }
       *r = NULL;
       return(++loc);
}

exist(s, c) char *s, c;  /* return 1 if c found in s, else 0 */
{
       while(*s)
	      if (*s++ == c) return(1);
       return(0);
}
 
/* assign value to keyword */

assnkey(s) char *s;
{
       char key[MAXKEYL], sub[MAXSUB];
       int loc;

       loc = token(s, key, 0, "= ", MAXKEYL);
       token(s, sub, loc, "= ", MAXSUB);
       addkey(key, sub);
}


condition(s) char *s; /* skip n lines if the given keyword is null or */
		      /* undefined.                                   */
{
       register loc, keycnt;
       char key[MAXKEYL], numb[10];
       int skipflg;

       loc = token(s, key, 0, "; ", MAXKEYL);
       token(s, numb, loc, "; ", 10);
       for(keycnt=0; keycnt<numkeys; keycnt++)
	      if (strcmp(keys[keycnt].keyword,key)==0)
		     break;
       if(keycnt<numkeys)
	      if(keys[keycnt].subst[0]==NULL)
		     skipflg = 1;
	      else
		     skipflg = 0;
       else
	      skipflg = 1;
       if (skipflg) {
	      skipline = atoi(numb);
	      if(skipline<=0) skipline = 1;
       }
}
 
/* read and write given file with no processing */

include(fn) char *fn;
{
       FILE *infile;
       int i;
       int len;

       if((infile=fopen(fn,"r"))==NULL) {
	      fprintf(stderr, "Cannot open %s\n", fn);
	      errflg++;
	      return;
       }

       while((len=getline(infile)) > 0) {
              if (tflag) fputs(line, stderr);
              if (lflag) fputs(line, stdout);
              if (len > MAXLINE) {
                         fprintf(stderr, "Line truncated: greater than %d characters.\n", MAXLINE);
			 len = MAXLINE;
                         errflg++;
              }
              if (rflag)
                     fwrite(line, len, 1, outfile);
              else {
                     padline();
                     xlate();
                     fwrite(line, MAXLINE, 1, outfile);
              }
       }

       fclose(infile);
       return;
}
 
/* read and write given file blocking into 80 byte records */

mod80(fn)
char *fn;
{
       FILE *infile;
       int bytes, cnt;
       int i;
       char *s;

       if((infile=fopen(fn,"r"))==NULL) {
	      fprintf(stderr, "Cannot open %s\n", fn);
	      return;
       }

       cnt = 0;
       while((bytes = fread(line, 1, LNL, infile)) > 0) {
              cnt += bytes;
              if (!rflag)
                     m80xlate(line, bytes);
              fwrite(line, bytes, 1, outfile);
       }

       if (cnt%80) {
              bytes = 80 - (cnt%80);
              while (bytes--){
	             putc('\0', outfile);
	      }
       }
       fclose(infile);
       return;
}


/* begin reading another file */

dofile(fn) char *fn;
{
       FILE *infile;

       if((infile=fopen(fn,"r"))==NULL) {
	      fprintf(stderr, "Cannot open %s\n", fn);
	      errflg++;
	      return;
       }
       process(infile);
       fclose(infile);
       return;
}
 
/*  'addkey' dynamically allocates storage to accommodate keyword
 *  definitions and their substitution values.
 */

addkey(k,s) char *k, *s;
{
    int keytotal, match;
    register int  ptr;
    register char *key,*sub;
    extern int revcpr();

    keytotal = numkeys-1;
    ptr = 0;
    while (keytotal >= 0)  {     /* check for duplicate keywords */
	   match = 1;
	   key = keys[keytotal].keyword;
	   while (*key && match)  {
		   if (k[ptr++] != *key++)
		       match = 0;
	   }
	   if (match && k[ptr] == NULL)  {
	      /* duplicate keyword ... reallocate space for new
	       * substitution string.
	       */
	       keys[keytotal].subst = malloc((unsigned)(strlen(s) + 1));
	       ptr = 0;
	       sub = keys[keytotal].subst;
	       while(*sub++ = s[ptr++]);
	       return;
	   }
	   keytotal--;
	   ptr = 0;
    }
    if (numkeys > MAXKEYS-1) {
	   fprintf(stderr, "Maximum keyword defines exceeded.\n");
	   errflg++;
	   return;
    }
   /* determine number of bytes required to store keyword and
    * substitution and request more core.
    */
    keys[numkeys].keyword = malloc((unsigned)(strlen(k) + 1));
    keys[numkeys].keylen  = strlen(k);
    keys[numkeys].subst   = malloc((unsigned)(strlen(s) + 1));
    key = keys[numkeys].keyword;
    sub = keys[numkeys].subst;
    ptr = 0;
    while(*key++ = k[ptr++]);
    ptr = 0;
    while(*sub++ = s[ptr++]);
    numkeys++;
    /*
     * as our final act we will sort the pointer structure in
     * reverse order by length of the keyword string so that keywords
     * which begin with the same characters aren't fouled up.
     */
    if (numkeys > 1)
	   qsort(keys, numkeys, 12, revcpr);
}
 
revcpr(m, n) int *m, *n; /* reverse compare for qsort of keys   */
{
	 if(m[2] < n[2]) return(1);
	 if(m[2] > n[2]) return(-1);
	 return(0);
}


/*  'donormal' takes an input line retrieved from a specified source
 *  file, calls a function to perform keyword substitution (if
 *  applicable), calls a function to fold alphabetic characters from
 *  lower to upper case and from ascii to ebcdic if specified,
 *  then writes the line to the temporary file created to
 *   collect the job/run stream.
 */

donormal()
{
    register i;

    if (!kflag && numkeys)
	mtchkeys();
    if(fflag)
	fold();
    if(tflag)
       fputs(line, stderr);   /* write input line to error output    */
    if(lflag)
       fputs(line, stdout);   /* write input line to standard output */
    padline();
    if(!rflag)
	xlate();
    fwrite(line, MAXLINE, 1, outfile);
}
 
/*  'fold' converts all alphabetic characters from lower to upper case
 */

fold()
{
    register char *i;

    i = &line[0];
    while (*i) {
	   *i = lowtup[*i];
	   i++;
    }
}

/*  'xlate' translates all characters from ascii to ebcdic
 */

xlate()
{
    register char *i;

    i = &line[0];
    while (*i) {
	   *i = atetab[*i];
	   i++;
    }
}

m80xlate(buf, cnt)
char    *buf;
int     cnt;
{
    register char *s;
    register int  i;

    s = buf;
    for (i = 0; i < cnt; i++) {
	   *s = atetab[*s];
	   *s++;
    }
}
 
/*  'padline' pads normal jcl lines with spaces to make them card
 *  images.  This must be done as the outfile is punched in raw mode.
 */

padline()
{
    register int i;

    i = strlen(line);
    if (i > MAXLINE) {
	  fprintf(stderr, "Line truncated: greater than %d characters.\n",MAXLINE);
	  errflg++;
    } else
	  for (; i < MAXLINE; i++)
                line[i] = ' ';
}
 
/*  'mtchkeys' scans an input string in the global buffer 'line' and
 *  attempts to match substrings of the line with the active set
 *  of keywords.  The new line with all substitutions is returned in
 *  the 'line' buffer.
 */

mtchkeys()
{
    register keycnt, loc;

    keycnt = 0;
    while (keycnt < numkeys) {
	    loc = 0;

	    /* KEEP LOOPING UNTIL WE CAN'T FIND THE KEYWORD          */
	    while ((loc=find(keys[keycnt].keyword, loc))>=0)
		    /* FOUND KEYWORD, DO SINGLE PASS SUBSTITUTION    */
		    loc=substitute(keys[keycnt].subst, loc,
				  strlen(keys[keycnt].keyword));
	    keycnt++;
    }
}

find(key,loc)
char *key;
int loc;
{
/*
 * Scan the line buffer starting at location loc for the given
 * keyword.  if found, return new loc, else return -1.
 */
        while(line[loc]) {
	        if (streq(key, &line[loc]))
                        return(loc);
	        loc++;
        }
        return(-1);
}
 
substitute(sub,loc,len) char *sub; int loc, len;
/*
 * substitute the given 'sub' string into 'line' at location loc.
 * ignore 'len' characters in 'line' at location loc.
 */
{
       register char *s, *t;
       register i;
       int newloc;

       for(i=0; i<loc; i++) wrkbuf[i] = line[i];
       s = sub;
       while(*s && i<LNL-2) wrkbuf[i++] = *s++;
       newloc = i;
       loc += len;
       while(line[loc] && i<LNL-2)
	      wrkbuf[i++] = line[loc++];
       wrkbuf[i] = NULL;
       s = &line[0];
       t = &wrkbuf[0];
       while(*s++ = *t++) ;
       return(newloc);
}
 
/*  'submit' invokes the process, 'vmpunch', to spool the temporary
 *  file of the job/run stream to CCCMVS
 */

submit()
{
    int  respond, ignore;
    char taginfo[40];
    int try, i;

    if (errflg)  {
	fprintf(stderr,"%d error%s detected - type 'y' to submit anyway: ",
	      errflg, errflg==1? "":"s");
	respond = ignore = getchar();
	while(ignore != NEWLINE && ignore != EOF)
	      ignore = getchar();
	fprintf(stderr, "\n");
	if (respond != 'y')  return;
    }
    for ( try=0; try<10; try++)  {
	  if ((i=fork()) >= 0) break;
	  sleep(2);
    }
    if (i < 0)  {
	fprintf(stderr,"Cannot fork for job submit.\n");
	cancel();
    }
    if (i == 0)  {    /* child process */
	sprintf(taginfo, "%s JOB tag=%s", complex, auuser);
        execl ( "/bin/vmpunch", "vmpunch", tempfil, "-r",
                "-d", "-v", OUTSYS, "-t", taginfo, 0);
        cancel();
    }
    exit(0);      /* parent */
}


/*  'cleanup' closes and unlinks the temporary output file and exits
 */

cleanup()
{
    fclose(outfile);
    unlink(tempfil);
    exit(040);
}

/* 'cancel' tells user that vmpunch failed and exits
 */

cancel()
{
    fprintf(stderr,"VMPUNCH failed for job ... JCL in file %s\n",tempfil);
    exit(1);
}
