/*
 * Description:
 *  These programs will generate the configuration files conf.c and
 *      iotab.c given a list of device addresses and types. The types
 *      are generally an IBM device type designation (ie. 1403 for a
 *      IBM 1403 or equivalent printer). A certain number of generic
 *      types are also accepted as equivalent to the usual devices of
 *      those types.
 *
 *  The list of accepted types is as follows:
 *      Terms in brackets are the generic types.
 *
 *      1052       (console)        - 1052 or equivalent console
 *      3215           "                        "
 *
 *      2305       (dasd)           - IBM 2305 or equivalent
 *      3330         "              - IBM 3330 or equivalent
 *      3340         "              - IBM 3340 or equivalent
 *      3350         "              - IBM 3350 or equivalent
 *
 *      3420       (tape)           - IBM 3420 tape drive or equivalent
 *
 *      2501       (reader)         - IBM 2501 card reader or equivalent
 *      2540r        "              - IBM 2540 card reader or equivalent
 *
 *      2540p      (punch)          - IBM 2540 card punch or equivalent
 *
 *      1403       (printer)        - IBM 1403 printer or equivalent
 *
 *  In addition, a certain number of "pseudo" device types are accepted.
 *      These are used as a convenient means of communicating with VM/470
 *      in general. The address for these devices is used to generate a
 *      device name/minor number, and should be in the range 0-n where n+1
 *      is the maximum of these devices to be generated.
 *
 *  Type specifiers for these pseudo devices are as follows:
 *
 *      vm_ldev                    - a logical device for VM.
 *      vm_smsg                    - a port into VM though which
 *                                   to send and receive messages to/from
 *                                   other virtual machines running under
 *                                   VM.
 *      vm_cpcmd                   - a port through which to send commands
 *                                   to CP and receive responses.
 *      vm_pvm                     - a port to the VM pass through virtual
 *                                   machine
 *      vio                        - an incore file system.
 */

#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "ioconf.h"

struct  device devices[MAXDEV];
struct  device *ndevp;             /* pointer to next available devicestructure */
struct  device *rtdev;             /* pointer to root device */

int    firstcall = 0;              /* first call indicator */
int    cmajor = 0;                 /* current major number for character devices */
int    bmajor = 0;                 /* current major number for block devies */
int    rootdev;                 /* major number for root disk */
int    pipedev;                 /* major number for pipe disk */
int    dumpdev;                 /* major number for dump disk */

char    devfile[] = "/newdev";
extern  char    *cmdname;
extern  struct  type    types[];
extern  struct  type    spectypes[];
extern  struct  driver  drivers[];
extern  struct  dmydrv  dmydrv[];
extern  struct  confhdr confhdr[];
extern  struct  spool   spdaems[];


/*
 *  dev(address, type)  - to accept a device to be configured
 *
 *       address  is the address for the device
 *       type     is a pointer to the device type.
 *
 *       returns - 0  ok
 *                 1  unknown type
 *                 2  invalid or duplicate address
 *                 3  directory access problems
 */
dev(address, type, mount)
int     address;
char    *type;
char    *mount;
{
        struct device **devpp;       /* pointer to pointer to device structure*/
        struct device *devp;         /* pointer to device structure*/
        struct type   *typp;         /* pointer into type table */
        struct driver *drvp;         /* pointer into driver table */
        struct dmydrv *dmy;          /* pointer into dummy driver table */
	struct device *finddev();
        int    devadded, dcount, error, fcount, i, k;
        char   name[30];
	char    *malloc();

        /*
         *  If this is the first call, see if directory "/newdev" exists.
         *     if so, print error message and quit,
         */

        if (firstcall == 0) {
                if (access(devfile, 0) == 0) {
                        fprintf(stderr, "%s: Directory /newdev already exists, remove it before retrying.\n", cmdname);
                        return(3);
                }
                firstcall = 1;
                ndevp = devices;    /* set pointer to first available
                                        device structure */
                umask(0);            /* make sure umask is right */
        }

        /*
         * check basic correctness of address
         */
        if (address > 4095) {
                fprintf(stderr, "%s: Invalid device address \"%3x\".\n", cmdname, address);
                return(2);
        }

        error = 0;
        fcount = 0;
        devadded = 0;              /* device not added yet */
        for (i = 0; ; i++) {
                typp = &types[i];
                if (typp->typtype == NULL)
                        break;
                if (strcmp(typp->typtype, type) == 0) {
                        fcount += 1;       /* increment found type count */
                        dcount = 0;        /* check to see if we find driver */
                        for (drvp = drivers; drvp->drvname != 0; drvp++) {
                                if (typp->typdriver == drvp->drvname) {
                                        dcount = 1;
                                        devpp = &drvp->drvdevice;
                                        devp = drvp->drvdevice;
                                        k = 0;
                                        while (devp != NULL) {
                                                devpp = &devp->devdevice;
                                                devp = devp->devdevice;
                                                k += 1;
                                        }
                                        if (*drvp->drvadtab != '\0') {   /* if its real device, add address */
                                                if (devadded != 1) {
                                                        genname(typp->typnmpfx, address, name);
                                                        if (adddev(address, name) !=0) {
                                                                error = 2;
                                                                break;
                                                        }
                                                        devadded = 1;     /* we have added the device */
                                                }
                                        }
                                        *devpp = ndevp;
                                        /*
                                         * fill in new device structure
                                         */
                                        ndevp->devdevice = NULL;
                                        ndevp->devtype = typp;
                                        ndevp->devaddress = address;
                                        ndevp->devtty = typp->typtty;
                                        if (mount != NULL) {
						ndevp->devmount = malloc(strlen(mount) + 1);
						strcpy(ndevp->devmount, mount);
                                        } else
                                                ndevp->devmount = NULL;
                                        drvp->drvnum++;
                                        if (*drvp->drvtype == 'c') {
                                                if (k == 0)     /* if this is the first use of this driver */
                                                        drvp->drvmajor = cmajor++;
                                                ndevp->devmajor = drvp->drvmajor;
                                                ndevp->devminor = k;
                                        } else {
                                                ndevp->devmajor = bmajor++;
                                                ndevp->devminor = 0;
                                        }
                                        ndevp++;  /* point at next available device structure */
                                        break;
                                }
                        }
                        if (dcount == 0) {
                                fprintf(stderr, "%s: Internal error - table inconsistencies\n", cmdname);
                                fprintf(stderr, "       driver type %d not found.\n", typp->typdriver);
                                return(3);
                        }
                        if (error != 0) {
                                fprintf(stderr, "%s: Duplicate address \"%3x\".\n", cmdname, address);
                                devadded = 1;          /* pretend it has been added */
                        }
                }
        }
	if (fcount == 0) {
                for (dmy = dmydrv; dmy->dmyname != NULL; dmy++) {
                        if (strcmp(dmy->dmyname, type) == 0) {
                                dmy->dmydevice = finddev(address);
                                if (dmy->dmydevice == NULL) {
					fprintf(stderr, "%s: Address %d for %s not previously defined\n", cmdname, address,type);
					error = 3;
                                }
                                fcount = 1;
			}
		}
	}

        if (fcount == 0) {
                fprintf(stderr, "%s: Unknown type \"%s\".\n", cmdname, type);
                error = 1;
        }
        return(error);
}

/*
 * genname(nmprfx, num, name) - routine to generate a device name
 *
 *     nmprfx    is a pointer to name string prefix of the form "c%%%",
 *               where c stands for zero or more characters. Each percent
 *               will be replaced by a hex digit from num.
 *     num       is the number of this device for name purposes.
 *     name      is a pointer to a place to build the name
 *
 */
genname(nmprfx, num, name)
char    *nmprfx;
int     num;
char    *name;
{

        int     i;
        char    format[10], *pn, *p;

        /*
         * first we build up the name from nmprfx and num
         */
        pn = name;
        for (p = nmprfx;(*p != '\0') && (*p != '%'); p++)
                *pn++ = *p;                /* copy the root part */
        for (i = 0; *p != '\0'; p++)
                i++;                       /* count number of percents */
        if (i > 0) {
                /*
                 *  Convert to hex, using specifed width
                 */
                sprintf(format, "%0*x", i, num);
                /*
                 * Copy to the output name, translate upper case hex
                 * digits to lowercase, since printf does it wrong.
                 */
                for (p=format;*p != '\0';p++)
                        if (isupper(*p))
                                *pn++ = tolower(*p);
                        else
                                *pn++ = *p;
        }
        *pn = '\0';         /* make sure of terminator */
        return;
}

/*
 * gendev(code, mode, major, minor, name) - routine to generate/newdev file.
 *
 *     code      is a pointer to type code either "c"  - character device
 *                                                "b"  - block device
 *                                                "d"  - directory
 *                                                "p"  - block dev used
 *                                                       for pipes.
 *     mode      the octal mode for the node
 *     major     is the major number to use for device
 *     minor     is the minor number to use for device
 *     name      is a ponter to the name for device
 */
gendev(code, mode, majorno, minorno, name)
char   *code;
int    mode;
int    majorno;
int    minorno;
char   *name;
{

        int     imode, numbers;
        char    devname[30];
        char    cmd[30];

        /*
         * first we build up the complete name
         */
	strcpy(devname, devfile);
	if (name[0] != '\0') {
		strcat(devname, "/");
		strcat(devname, name);
	}
        /*
         * now figure out mode bits
         */
	switch(*code) {
	case 'c':
                imode = S_IFCHR | mode;
		break;
	case 'b':
                imode = S_IFBLK | mode;
		break;
	case 'd':
                imode = S_IFDIR | mode;
		break;
	}

        if (*code != 'd') {
                numbers = (majorno << 8) + minorno;  /*  compute major/mi nor number */
                if (mknod(devname, imode, numbers) != 0)    /* make the s pecial file */
                        fprintf(stderr, "%s: Unable to create %s\n", cmdname, devname);
        } else {
		sprintf(cmd, "mkdir %s", devname);
		if (system(cmd) != 0)
                        fprintf(stderr, "%s: Unable to mkdir %s\n", cmdname, devname);
		chmod(devname, imode);
	}
        return;
}

/*
 *  genspec - generate the always present special stuff in conf.c
 */
genspec()
{
        struct  driver  *drvp;
        struct  type    *typp;
        struct  device  *devp, **devpp;
        int     dcount, k;

        for (typp = spectypes; typp->typdriver != 0; typp++) {
                dcount = 0;        /* check to see if we find driver */
                for (drvp = drivers; drvp->drvname != 0; drvp++) {
                        if (typp->typdriver == drvp->drvname) {
                                dcount = 1;
                                devpp = &drvp->drvdevice;
                                devp = drvp->drvdevice;
                                k = 0;
                                while (devp != NULL) {
                                         devpp = &devp->devdevice;
                                         devp = devp->devdevice;
                                         k += 1;                       /* and count devices present */
                                }
                                *devpp = ndevp;
                                /*
                                 * fill in new device structure
                                 */
                                ndevp->devdevice = NULL;
                                ndevp->devtype = typp;
                                ndevp->devaddress = 0;
                                ndevp->devtty = "";   /* null for special devices */
                                if (*drvp->drvtype == 'c') {
                                        if (k == 0)     /* if this is the first use of this driver */
                                                drvp->drvmajor = cmajor++;
                                        ndevp->devmajor = drvp->drvmajor;
                                } else
                                        ndevp->devmajor = bmajor++;
                                ndevp->devminor = atoi(typp->typtty);
                                ndevp++;  /* point at next available device structure */
                                break;
                        }
                }
                if (dcount == 0) {
                        fprintf(stderr, "%s: Internal error1 - table inconsistencies\n", cmdname);
                        fprintf(stderr, "       driver type %d not found.\n", typp->typdriver);
                        exit(1);
                }
        }
}

/*
 * Generate any directories necessary
 */
gendirs()
{
        struct  driver  *drvp;
        struct  type    *typp;
        struct  device  *devp;
	char    name[30];

        for (drvp = drivers; drvp->drvname != 0; drvp++) {
                if ((*drvp->drvtype != 'd') || (drvp->drvdevice == NULL))
                        continue;
                for (devp = drvp->drvdevices; devp != NULL; devp = devp->devdevice) {
                        typp = devp->devtype;
                        genname(typp->typnmpfx, devp->devaddress, name);
                        gendev(drvp->drvtype, typp->typmode, devp->devmajor, devp->devminor, name);
                }
        }
}

/*
 *  genconf - generate conf.c
 */
genconf()
{
        static  char    conffile[] = "/usr/src/sys/conf/newconf.c";
        static  char    ttysfile[] = "/etc/newttys";
        FILE    *cofp,  *ttyfp;
        struct  driver  *drvp;
        struct  type    *typp;
        struct  device  *devp;
        int     i;
        char    name[30];

        if ((cofp = fopen(conffile, "w")) == NULL) {
                fprintf(stderr, "%s: Unable to open %s.\n", cmdname, conffile);
                exit(1);
        }
        if ((ttyfp = fopen(ttysfile, "w")) == NULL) {
                fprintf(stderr, "%s: Unable to open %s.\n", cmdname, ttysfile);
                exit(1);
        }
        fprintf(cofp, "#include \"../h/param.h\"\n");   /* first header file */
        fprintf(cofp, "#include \"../h/systm.h\"\n");   /* stuff */
        fprintf(cofp, "#include \"../h/buf.h\"\n");
        fprintf(cofp, "#include \"../h/tty.h\"\n");
        fprintf(cofp, "#include \"../h/conf.h\"\n");
        fprintf(cofp, "#include \"../h/proc.h\"\n");
        fprintf(cofp, "#include \"../h/dir.h\"\n");
        fprintf(cofp, "#include \"../h/user.h\"\n");
        fprintf(cofp, "#include \"../h/text.h\"\n");
        fprintf(cofp, "#include \"../h/file.h\"\n");
        fprintf(cofp, "#include \"../h/inode.h\"\n\n");

        fprintf(cofp, "typedef int func();\n\n");       /* function definition */

        /*
         * Generate bdevsw and associated stuff
         */
        fprintf(cofp, "/*\n");                      /* now some comments */
        fprintf(cofp, " * the block device table, bdevsw\n");
        fprintf(cofp, " */\n\n");

	/*
	 *  generate entry point declarations
	 */
        for (drvp = drivers; drvp->drvname != 0; drvp++) {
                if ((*drvp->drvtype == 'b') && (drvp->drvdevice != NULL))
                        fprintf(cofp, "func   %s;\n", drvp->drveps);
        }
        fprintf(cofp, "\n");

        /*
         *   generate  buffer structures needed by block devices
         */

        for (drvp = drivers; drvp->drvname != 0; drvp++) {
                if ((*drvp->drvtype != 'b') || (drvp->drvdevice == NULL))
                        continue;
                for (devp = drvp->drvdevices; devp != NULL; devp = devp->devdevice) {
                        typp = devp->devtype;
                        genname(typp->typnmpfx, devp->devaddress, name);
                        fprintf(cofp, "struct buf %stab;\n", name);
                }
        }
        fprintf(cofp, "\n");       /* some white space */

        /*
         *  generate  bdevsw structure
         */
        fprintf(cofp, "struct bdevsw bdevsw[] = {\n");
        for (drvp = drivers; drvp->drvname != 0; drvp++) {
                if ((*drvp->drvtype != 'b') || (drvp->drvdevice == NULL))
                        continue;
                for (devp = drvp->drvdevices; devp != NULL; devp = devp->devdevice) {
                        typp = devp->devtype;
                        genname(typp->typnmpfx, devp->devaddress, name);
                        gendev(drvp->drvtype, typp->typmode, devp->devmajor, 0, name);
                        fprintf(cofp, "        %s, &%stab, 0x%3x, \n",
                                drvp->drveps, name, devp->devaddress);
                }
        }
        fprintf(cofp, "        0\n};\n\n");

	genroot();
        /*
         *  Generate cdevsw and associated stuff
         */
        fprintf(cofp, "/*\n");
        fprintf(cofp, " * the character device table, cdevsw\n");
        fprintf(cofp, " */\n\n");

        for (drvp = drivers; drvp->drvname != 0; drvp++) {
                if ((*drvp->drvtype == 'c') && (drvp->drvdevices != NULL))
                        fprintf(cofp, "func   %s;\n", drvp->drveps);
        }
        fprintf(cofp, "\n");      /* some white space */

        /*
         * generate address lists
         */
        for (drvp = drivers; drvp->drvname != 0; drvp++) {
                if ((*drvp->drvtype != 'c') || (drvp->drvdevice == NULL) || (*drvp->drvadtab == '\0'))
                        continue;

                fprintf(cofp, "int     %s[] = {", drvp->drvadtab);
                for (devp = drvp->drvdevices; devp != NULL; devp = devp->devdevice) {
                        fprintf(cofp, "0x%3x, ", devp->devaddress);
                }
                fprintf(cofp, "};\n");
        }
        fprintf(cofp, "\n");     /* some white space */

        /*
         *  generate  cdevsw structure
         *     which must be sorted in major number order
         */
        fprintf(cofp, "struct cdevsw cdevsw[] = {\n");
        for (i = 0; ; i++) {
                drvp = &drivers[i];
                if (drvp->drvname == 0)
                        break;
                for (drvp = drivers; drvp->drvname != 0; drvp++) {
                        if ((*drvp->drvtype != 'c') || (drvp->drvdevice == NULL) || (drvp->drvmajor != i))
                                continue;
                        fprintf(cofp, "        %s, ", drvp->drveps);
                        if (*drvp->drvadtab != '\0')
                                fprintf(cofp, "%s, 0, \n", drvp->drvadtab);
                        else
                                fprintf(cofp, "0, 0, \n");
                        for (devp = drvp->drvdevices; devp != NULL; devp = devp->devdevice) {
                                typp = devp->devtype;
                                genname(typp->typnmpfx, devp->devaddress, name);
                                gendev(drvp->drvtype, typp->typmode, devp->devmajor, devp->devminor, name);
                                if (*devp->devtty != '\0')
                                        fprintf(ttyfp, "%s%s\n", devp->devtty, name);
                        }
                        break;
                }
        }
        fprintf(cofp, "        0\n};\n\n");

        fprintf(cofp, "int     ttyopen(), ttyclose(), ttread(), ttyinput(), ttstart();\n");
        fprintf(cofp, "char    *ttwrite();\n");
        fprintf(cofp, "struct  linesw linesw[] = {\n");
        fprintf(cofp, "        ttyopen, nulldev, ttread, ttwrite, nodev, ttyinput, ttstart, \n");
        fprintf(cofp, "        0\n};\n\n");
        fprintf(cofp, "dev_t   rootdev = makedev(%d, 0);\n", rootdev);
        fprintf(cofp, "dev_t   pipedev = makedev(%d, 0);\n", pipedev);
        fprintf(cofp, "dev_t   dumpdev = makedev(%d, 0);\n\n", dumpdev);
        fprintf(cofp, "int     nldisp = 1;\n\n");
        fprintf(cofp, "struct  buf     buf[NBUF];\n");
        fprintf(cofp, "struct  buf     bfreelist;\n");
        fprintf(cofp, "struct  proc    proc[NPROC];\n");
        fprintf(cofp, "struct  inode   inode[NINODE];\n");
        fprintf(cofp, "struct  file    file[NFILE];\n");
        fprintf(cofp, "struct  text    text[NTEXT];\n");
        fprintf(cofp, "int     mpxchan();\n");
        fprintf(cofp, "int     (*ldmpx)() = mpxchan;\n");

        fclose(cofp);
        fclose(ttyfp);
	return(0);
}
/*
 *  genmtab - generate normal name for mtab file
 */
genmtab()
{
	FILE    *mtabfp;
	struct  type    *typp;
        static  char    mtabline[64] = "/";
        static  char    mtabfile[] = "/etc/newrootmtab";

        if ((mtabfp = fopen(mtabfile, "w")) == NULL) {
                fprintf(stderr, "%s: Unable to open %s.\n", cmdname, mtabfile);
                exit(1);
        }
	typp = rtdev->devtype;
        genname(typp->typnmpfx, rtdev->devaddress, &mtabline[32]);
        fwrite(mtabline, 1, sizeof(mtabline), mtabfp);
	fclose(mtabfp);
}

/*
 *  genroot - generate rootdev stuff
 *  Define the major devices numbers for the dummy
 *  device entries.  If the root device is not specified,
 *  use the first dasd.  If the pipe device is not specified,
 *  use the first vio.  If the dump device is not specified,
 *  use the root device.
 */
genroot()
{
	struct  dmydrv  *dmyp;
	struct  device  *firstdev();

	for (dmyp = dmydrv; dmyp->dmyname != NULL; dmyp++) {
		switch(dmyp->dmytype) {
		case D_ROOT:
                        if (dmyp->dmydevice == NULL) {
                                dmyp->dmydevice = firstdev(D_DASD);
				if (dmyp->dmydevice == NULL) {
					fprintf(stderr, "%s: No dasd devices\n", cmdname);
                                        exit(1);
				}
			}
			rootdev = dmyp->dmydevice->devmajor;
                        rtdev = dmyp->dmydevice;
			break;
		case D_PIPE:
			if (dmyp->dmydevice == NULL) {
				dmyp->dmydevice = firstdev(D_VIO);
				if (dmyp->dmydevice == NULL)
					dmyp->dmydevice = rtdev;
			}
			pipedev = dmyp->dmydevice->devmajor;
			break;
		case D_DUMP:
			if (dmyp->dmydevice == NULL)
				dmyp->dmydevice = rtdev;
			dumpdev = dmyp->dmydevice->devmajor;
			break;
		}
	}
}
struct device *
firstdev(type)
int     type;
{
	struct  driver  *drvp;

        for (drvp = drivers; drvp->drvname != NULL; drvp++) {
                if (drvp->drvname == type)
			break;
        }
        return(drvp->drvdevice);
}

/*
 *  genconfigs - generate nioreconfig.cp and niounconfig.cp.
 *  If it is a terminal, 3270, or tape, put a detach line in
 *  niounconfig.cp.
 *  If it is a terminal or 3270 put a define line
 *  in nioreconfig.cp and be sure to include some fancy links to
 *  trick vm into giving us a nice controller.
 */
genconfigs()
{
	struct  driver  *drvp;
	struct  device  *devp;
	struct  type    *typp;
	int     drvr;
	static  char    niounfile[] = "/etc/niounconfig.cp";
	static  char    niorefile[] = "/etc/nioreconfig.cp";
        FILE    *nioun;
        FILE    *niore;


        if ((nioun = fopen(niounfile, "w")) == NULL) {
                fprintf(stderr, "%s: Unable to open %s.\n", cmdname, niounfile);
                exit(1);
        }
        if ((niore = fopen(niorefile, "w")) == NULL) {
                fprintf(stderr, "%s: Unable to open %s.\n", cmdname, niorefile);
                exit(1);
        }
        for (drvp = drivers; drvp->drvname != 0; drvp++) {
                if ((devp = drvp->drvdevice) == NULL)
			continue;
                typp = devp->devtype;
                drvr = typp->typdriver;

                if (drvr == D_TERM) {
                        fprintf(nioun, "*\n");
                        fprintf(nioun, "*  Detach terminals when entering single user mode\n");
                        fprintf(nioun, "*\n");
                        fprintf(nioun, "det %3x\n", devp->devaddress);

                        fprintf(niore, "*\n");
                        fprintf(niore, "*  link gets vm to give us a nice controller\n");
                        fprintf(niore, "*\n");
                        fprintf(niore, "link * %3x %3x\n", rtdev->devaddress, devp->devaddress + 1);
                        fprintf(niore, "def line %3x tele2\n", devp->devaddress);
                        fprintf(niore, "det %3x\n", devp->devaddress + 1);

                        while ((devp = devp->devdevice) != NULL) {
                                fprintf(nioun, "det %3x\n", devp->devaddress);
                                fprintf(niore, "def line %3x tele2\n", devp->devaddress);
                        }

                } else if (drvr == D_TUBE) {
                        fprintf(nioun, "*\n");
                        fprintf(nioun, "*  Detach terminals when entering single user mode\n");
                        fprintf(nioun, "*\n");
                        fprintf(nioun, "det %3x\n", devp->devaddress);

                        fprintf(niore, "*\n");
                        fprintf(niore, "*  link gets vm to give us a nice controller\n");
                        fprintf(niore, "*\n");
                        fprintf(niore, "link * %3x %3x\n", rtdev->devaddress, devp->devaddress + 1);
                        fprintf(niore, "def graf %3x\n", devp->devaddress);
                        fprintf(niore, "det %3x\n", devp->devaddress + 1);

                        while ((devp = devp->devdevice) != NULL) {
                                fprintf(niore, "def graf %3x\n", devp->devaddress);
                                fprintf(nioun, "det %3x\n", devp->devaddress);
                        }

                } else if (drvr == D_TAPE) {
                        fprintf(nioun, "*\n");
                        fprintf(nioun, "*  Detach tapes in case any were mounted\n");
                        fprintf(nioun, "*\n");
                        fprintf(nioun, "det %3x\n", devp->devaddress);

                        while ((devp = devp->devdevice) != NULL)
                                fprintf(nioun, "det %3x\n", devp->devaddress);
                }
	}
	fclose(nioun);
	fclose(niore);
	return(0);
}

/*
 *  genheader - generate the header file newioconf.h
 *  This header includes the constants define the number
 *  of various types of devices.
 */
genheader()
{
	struct  driver  *drvp;
        struct  device  *devp;
	struct  type    *typp;
	struct  confhdr *cf;
	FILE    *confh;
	int     drvr;
	static  char    ioconfh[] = "/usr/src/sys/h/newioconf.h";

	if ((confh = fopen(ioconfh, "w")) == NULL) {
		fprintf(stderr, "%s: unable to open %s\n", cmdname, ioconfh);
		exit(1);
	}

        for (drvp = drivers; drvp->drvname != 0; drvp++) {
                if ((devp = drvp->drvdevice) == NULL)
			continue;
                if ((typp = devp->devtype) == NULL)
			continue;
		drvr = typp->typdriver;

		for (cf = confhdr; cf->c_driver != NULL; cf++) {
			if (drvr == cf->c_driver) {
                                if (drvp->drvnum > 0)
                                        cf->c_num = drvp->drvnum;
				break;
			}
		}
	}
        for (cf = confhdr; cf->c_driver != NULL; cf++)
                fprintf(confh, "#define %s\t%d\t\t%s\n",
                        cf->c_define, cf->c_num, cf->c_comment);
	fprintf(confh, "#define NRDASD\tNDASD\t\t/* Raw Dasd */\n");
	fprintf(confh, "#define NVMC\t(NVMCF/2)\t/* VM Communication Facility devices */\n");
	fclose(confh);
	return(0);
}
/*
 * genmount - generate the mountlist file
 * This file contains the mount commands to mount all the
 * disks and is called by /etc/rc.
 */
genmount()
{
	struct  driver  *drvp;
        struct  device  *devp;
	struct  type    *typp;
	FILE    *mntfp;
	int     drvr;
	char    name[30];
	static  char    mntlist[] = "/etc/newmountlist";

	if ((mntfp = fopen(mntlist, "w")) == NULL) {
		fprintf(stderr, "%s: unable to open %s\n", cmdname, mntlist);
		exit(1);
	}

        for (drvp = drivers; drvp->drvname != 0; drvp++) {
                if ((devp = drvp->drvdevice) == NULL)
			continue;
                if ((typp = devp->devtype) == NULL)
			continue;
		drvr = typp->typdriver;
		if ((drvr != D_DASD) && (drvr != D_VIO))
			continue;

		do {
			if ((devp->devmount != NULL) &&
			    (devp->devaddress != rtdev->devaddress)) {
				typp = devp->devtype;
                                genname(typp->typnmpfx, devp->devaddress, name);
                                fprintf(mntfp, "/etc/mount /dev/%s %s\n", name, devp->devmount);
			}
                } while ((devp = devp->devdevice) != NULL);
	}
	fclose(mntfp);
	return(0);
}

/*
 *  genspool - generate the spooler list
 */
genspool()
{
	struct  driver  *drvp;
        struct  device  *devp;
	struct  type    *typp;
	struct  spool   *sp;
	FILE    *spfp;
	int     drvr;
	char    name[30];
	static  char    spoollist[] = "/etc/spooler/newspoollist";

	if ((spfp = fopen(spoollist, "w")) == NULL) {
		fprintf(stderr, "%s: unable to open %s\n", cmdname, spoollist);
		exit(1);
	}

        for (drvp = drivers; drvp->drvname != 0; drvp++) {
                if ((devp = drvp->drvdevice) == NULL)
			continue;
                if ((typp = devp->devtype) == NULL)
			continue;
		drvr = typp->typdriver;
		for (sp = spdaems; sp->sp_drvtype != 0; sp++) {
			if (drvr == sp->sp_drvtype)
				sp->sp_dev = devp;
                }
	}
        for (sp = spdaems; sp->sp_drvtype != 0; sp++) {
                for (devp = sp->sp_dev; devp != NULL; devp = devp->devdevice) {
                        typp = devp->devtype;
                        genname(typp->typnmpfx, devp->devaddress, name);
			fprintf(spfp, "%s %s /dev/%s\n", sp->sp_key, sp->sp_daem, name);
		}
	}
	fclose(spfp);
}
struct device *
finddev(addr)
int     addr;
{
	struct  device  *devp;

	for (devp = devices; devp < ndevp; devp++) {
		if (addr == devp->devaddress)
			return(devp);
	}
	return(NULL);
}
