/*
 * Driver to commmunicate with PVM virtual machine
 * Uses VMCF
 */
#include "../h/param.h"
#include "../h/systm.h"
#include "../h/conf.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/vmcf.h"
#include "../h/buf.h"
#include "../h/tty.h"

#define min(a, b) (a < b ? a : b)

/*
 * VMCF message IDs - one word containing 4 EBCDIC characters
 */
#define SIGNON  0xE2C9C7D5      /* 'SIGN' - sign-on or reconnect */
#define SIGNOFF 0xE2D5F040      /* 'SN0 ' - sign-off */
#define DATU    0xC4E4F040      /* 'DU0 ' - write data */
#define DATL    0xC4D3F040      /* 'DL0 ' - read-modified data */
#define DATB    0xC4C2F040      /* 'DB0 ' - read-buffered data */
#define DOWN    0xC4D5F040      /* 'DN0 ' - link down */
#define MSG     0xD4E2C740      /* 'MSG ' - message */
#define TDISC   0xE3D4F040      /* 'TM0 ' - temporary disconnect (?) */
#define WAKE    0xE6C1D2C5      /* 'WAKE' - wakeup (?) */

/* sign-on request identifier */
#define NETW    0xD5C5E3E6      /* 'NETW' */
/* sign-on confirmation message */
static char upmsg[] = "LINK SUCCESSFULLY ESTABLISHED";

#define READMOD 0x06            /* read-modified ccw */

struct req {
	int     fld;
	int     senlw;          /* session disc len */
	char    sgnda[8];       /* session disc data */
	char    node[8];        /* specific node */
	short   port;           /* specific port */
	short   real;           /* real terminal addr */
	short   class;          /* terminal class */
	char    model;          /* terminal model */
};

struct pvm {
	int     vd;             /* VMCF desc */
	int     flag;           /* flag bits */
	int     len;            /* input length */
	caddr_t ibuf;           /* input buffer */
	caddr_t obuf;           /* output buffer */
	struct req sgn;         /* signon request */
	int     pksig;          /* signal # for pokes */
	struct proc *pkproc;    /* process # for pokes */
} pvm;

/* flag bits */
#define OPEN    01
#define PEND    02      /* input pending */
#define ISDOWN  04      /* link is down */
#define INIT    010     /* needs sign-on request */
#define ISINIT  020     /* sign-on in progress */

extern char atetab[];
extern char etatab[];

/*
 * Open PVM connection
 */
/* ARGSUSED */
pvmopen(dev, flag)
{
	int pvmintr(), i;

	if(pvm.flag & OPEN) {
		u.u_error = EBUSY;
		return;
	}
	pvm.flag = OPEN|INIT;
	pvm.pksig = 0;
	pvm.ibuf = getpage();
	pvm.obuf = getpage();
	pvm.vd = vmc_catch("PVM", pvmintr, 0);
	vmc_canc(pvm.vd, SIGNON);
	vmc_canc(pvm.vd, DATL);
	vmc_canc(pvm.vd, DATB);
	vmc_rjct(pvm.vd, DATU);
	vmc_rjct(pvm.vd, DOWN);
	vmc_rjct(pvm.vd, SIGNON);
	vmc_rjct(pvm.vd, MSG);
	pvm.sgn.fld = NETW;
	pvm.sgn.senlw = 0;
	for(i=0; i<8; i++)
		pvm.sgn.sgnda[i] = atetab[' '];
	for(i=0; i<8; i++)
		pvm.sgn.node[i] = atetab[' '];
	pvm.sgn.port = 0xFF00;
	pvm.sgn.real = 0;
	pvm.sgn.class = 0x4004; /* 3277 */
	pvm.sgn.model = 2;      /* model 2 */
}

/*
 * Close PVM connection
 */
/* ARGSUSED */
pvmclose(dev)
{
	vmc_send(pvm.vd, SIGNOFF, pvm.obuf, 0);
	vmc_drop(pvm.vd);
	freepag(pvm.ibuf);
	freepag(pvm.obuf);
	pvm.flag = 0;
}

/*
 * Read from PVM
 */
/* ARGSUSED */
pvmread(dev)
{
	int n;

	if(pvm.flag & INIT)
		pvminit();
	if(pvm.flag & ISDOWN)
		return;
	while((pvm.flag & (PEND|ISDOWN)) == 0)
		sleep((caddr_t)&pvm.flag, TTIPRI);
	pvm.flag &= ~PEND;
	n = min(pvm.len, u.u_count);
	n = min(n, PSIZE);
	if(pvm.flag & ISDOWN)
                vmc_recv(pvm.vd, DOWN, pvm.ibuf, n);
	else
                vmc_recv(pvm.vd, DATU, pvm.ibuf, n);
	if(pvm.flag & ISDOWN)
		passc(0);
	iomove(pvm.ibuf, n, B_READ);
}

/*
 * Write to PVM
 */
/* ARGSUSED */
pvmwrite(dev)
{
	int n;

	if(pvm.flag & INIT)
		pvminit();
	if(pvm.flag & ISDOWN)
		return;
	n = min(u.u_count, PSIZE);
	iomove(pvm.obuf, n, B_WRITE);
	if(pvm.obuf[0] == READMOD)
		vmc_send(pvm.vd, DATL, pvm.obuf+1, n-1);
	else
		vmc_send(pvm.vd, DATB, pvm.obuf+1, n-1);
}

/*
 * Issue PVM sign-on request
 * Done on first read or write after open
 */
pvminit()
{
	int resid, n;
	char *cp1, *cp2;

	while(pvm.flag & ISINIT)
		sleep((caddr_t)&pvm.flag, TTIPRI);
	if((pvm.flag & INIT) == 0)
		return;
	pvm.flag |= ISINIT;
	if(vmc_senr(pvm.vd, SIGNON, (caddr_t)&pvm.sgn, 31, pvm.ibuf, PSIZE, &resid)) {
		u.u_error = EIO;
		pvm.flag &= ~(ISINIT|INIT);
		pvm.flag |= ISDOWN;
		printf("pvm sign-on failed\n");
		return;
	}
	n = PSIZE - resid;
	cp1 = pvm.ibuf;
	while(n--) {
		*cp1 = etatab[*cp1];
		cp1++;
	}
	*cp1 = '\0';
	printf("pvm: %s\n", pvm.ibuf);
        pvm.flag &= ~(ISINIT|INIT);
	wakeup((caddr_t)&pvm.flag);
	cp1 = pvm.ibuf;
	cp2 = upmsg;
	while(*cp2) if(*cp1++ != *cp2++) {
		u.u_error = EIO;
		pvm.flag |= ISDOWN;
		return;
	}
}

/*
 * ioctl for PVM
 * get or set sign-on parms
 */
/* ARGSUSED */
pvmioctl(dev, cmd, argp, flag)
caddr_t argp;
{
	switch(cmd) {

	case PVMGET:
		if(copyout((caddr_t)&pvm.sgn, argp, sizeof pvm.sgn))
			u.u_error = EFAULT;
		break;

	case PVMSET:
		if(copyin(argp, (caddr_t)&pvm.sgn, sizeof pvm.sgn))
			u.u_error = EFAULT;
		break;

	case TIOPOLL:
		suword(argp, (pvm.flag & (PEND|ISDOWN)) ? 1 : 0);
		break;

	case TIOPOKE:
		pvm.pksig = (int)argp;
		pvm.pkproc = u.u_procp;
		break;

	case TIONPOKE:
		pvm.pksig = 0;
		break;

	default:
		u.u_error = ENOTTY;
		break;
	}
}

/*
 * VMCF interrupt from PVM machine
 */
/* ARGSUSED */
pvmintr(arg, h)
register struct vmcmhdr *h;
{
	if(h->m_stat & (M_RESP|M_RJCT))
		return(1);
	if(h->m_mid == DATU) {
		pvm.flag |= PEND;
		pvm.len = h->m_lena;
		wakeup((caddr_t)&pvm.flag);
		if(pvm.pksig > 0) {
			psignal(pvm.pkproc, pvm.pksig);
			pvm.pksig = 0;
		}
		return(1);
	}
	if(h->m_mid == DOWN) {
		pvm.flag |= ISDOWN;
		pvm.len = h->m_lena;
		wakeup((caddr_t)&pvm.flag);
		if(pvm.pksig > 0) {
			psignal(pvm.pkproc, pvm.pksig);
			pvm.pksig = 0;
		}
		return(1);
	}
	return(0);
}
